# This script is part 2 of the pipeline; It:
# 1. Imports Quality filtered data
# 2. performs Sensor Self heating Correction following the method 4 of Burba et al. (2008) and the revised formulations following Frank & Massman (2020)
# 3. exports the results as a *_openeddy_SSH.csv for further processing
# 
# Felix Nieberding
# 2020-08-26


rm(list=ls())
Sys.setenv(TZ='UTC')

library(tidyverse)
library(lubridate)
library(data.table)
library(ggpmisc)
library(openeddy)
library(gridExtra)
library(xtable)


# select the respective dataset
dataset <- "ORG"
# dataset <- "WLG"

setwd(dir = "~/TransTiP/_NamCo_sync/3_data_qa-qc/Flux data/ESSD_scripts_review/")

# labeller
month_name <- c("1" = "January", "2" = "February", "3" = "March", "4" = "April", "5" = "May", "6" = "June", 
                "7" = "July", "8" = "August", "9" = "September", "10" = "October", "11" = "November", "12" = "December")

# custom constants --------------------------------------------------------
TEMP_THR = 0     # Temperature (°C) below which to apply the Burba correction

# Import data -------------------------------------------------------------
# import quality filtered data 
df_openeddy <- fread(paste("df_", dataset, "_openeddy.csv", sep = "")) %>% mutate(DATETIME = as.POSIXct(DATETIME))

# generate input file, take care to have the right units --------------------------------
df_SSH_input <- 
  df_openeddy %>%
  transmute(DATETIME = DATETIME,
            DOY_TIME = yday(DATETIME) + as.numeric(paste(str_sub(as.character(DATETIME), 12, 13), str_sub(as.character(DATETIME), 15, 16),sep = "")) / 2400,
            DATE = as_date(DATETIME),
            DOY = yday(DATETIME),
            TIME = as.numeric(paste(str_sub(as.character(DATETIME), 12, 13), str_sub(as.character(DATETIME), 15, 16), sep = "")) / 2400,
            TIME_DEC = as.numeric(str_sub(as.character(DATETIME), 12, 13)) + as.numeric(str_sub(as.character(DATETIME), 15, 16)) / 60,
            DAYTIME = daytime,
            P =  air_pressure / 1000, # (kPa) Air Pressure
            F_CO2 = ifelse(qc_NEE_composite == 2, NA, co2_flux), # (µmol/m²s) Quality filtered co2_flux
            F_H2O = ifelse(qc_h2o_composite == 2, NA, h2o_flux), # (mmol/m²s) Quality filtered h2o_flux
            CO2 = ifelse(qc_co2_molar_density_composite == 2, NA, co2_molar_density * 44), # (mg/m³) 
            H2O = ifelse(qc_h2o_molar_density_composite == 2, NA, h2o_molar_density * 0.018), # (g/m³) 
            TEMP_C = air_temperature -273.15, # (°C) Air Temperature
            MEAN_U = wind_speed, # (m/s) Wind speed
            RHOCP = air_density * air_heat_capacity, # (J/m³K) Rho is the mean total air  mass density (kg/m³) * Cp is the specific heat of air (J kg-1 K-1)
            PD = air_density * 1000) #(g/m³) Dry air density

# Burba correction --------------------------------------------------------
# calculations for Burba correction
SSH_correction_Burba <- function(df_input, RAD_THR, K) {
  df_input %>%
    mutate(TEMP_K = TEMP_C + 273.15,                                                                                         # Air Temperature in K
           K_AIR =  0.000067 * TEMP_C + 0.024343,                                                                            # (W/mK) thermal conductivity coefficient of air
           L_TOP =  0.045,                                                                                                   # (m) Diameter of detector housing
           L_BOT =  0.065,                                                                                                   # (m) Diameter of source housing
           L_SPAR = 0.005,                                                                                                   # (m) Diameter of the spar
           D_TOP =  0.0028 * sqrt(L_TOP / MEAN_U) + 0.00025 / MEAN_U + 0.0045,                                               # (m) Average thickness of boundary layer below top window
           D_BOT =  0.004  * sqrt(L_BOT / MEAN_U) + 0.004,                                                                   # (m) Average thickness of boundary layer above bottom window
           D_SPAR = 0.0058 * sqrt(L_SPAR / MEAN_U),                                                                          # (m) Average thickness of boundary layer around the spar
           T_TOP =  ifelse(DAYTIME == 1, (1.005 * TEMP_C + 0.24), (1.008 * TEMP_C - 0.41)) + 273.15,                         # (K) Surface temperature of top sensor housing (Burba, 2008), smoothed transition between night and day using tanh(K*RAD)
           T_BOT =  ifelse(DAYTIME == 1, (0.944 * TEMP_C + 2.57), (0.883 * TEMP_C + 2.17)) + 273.15,                         # (K) Surface temperature of bottom sensor housing (Burba, 2008), smoothed transition between night and day using tanh(K*RAD)
           T_SPAR = ifelse(DAYTIME == 1, (1.010 * TEMP_C + 0.36), (1.010 * TEMP_C - 0.17)) + 273.15,                         # (K) Surface temperature of spar (Burba, 2008), smoothed transition between night and day using tanh(K*RAD)
           S_TOP =  K_AIR * (((T_TOP - TEMP_K)  * (0.0225 + D_TOP)) / (0.0225 * D_TOP)),                                     # Sensible heat flux from top sensor housing 
           S_BOT =  K_AIR *  ((T_BOT - TEMP_K)  / D_BOT),                                                                    # (W/m²) Sensible heat flux from bottem sensor housing 
           S_SPAR = K_AIR *  ((T_SPAR - TEMP_K) / (0.0025 * log((0.0025 + D_SPAR) / 0.0025))),                               # (W/m²) Sensible heat flux from spar 
           HC =     (S_BOT + S_TOP + 0.15 * S_SPAR) / RHOCP * CO2 / TEMP_K * (1 + 1.6077 * H2O / PD),                        # (mg/m²/s) heating correction, add sensible heat fluxes from instrument surfaces to ambient sensible heat flux (eq. 4, Burba 2008), add weights from eq. 14, Frank & Massmann (2020)
           F_CO2 =  F_CO2 * 0.044,                                                                                           # (mg/m²s) F_CO2 before heating correction
           F_H2O =  F_H2O * 18,                                                                                              # (mg/m²s) F_H2O before heating correction
           F_CO2_HC = ifelse(TEMP_C < TEMP_THR, F_CO2 + HC, F_CO2),                                                          # (mg/m²s) Apply heating correction to CO2 flux when ambient temperature < TEMP_THR3
           F_H2O_HC = ifelse(TEMP_C < TEMP_THR, F_H2O + HC, F_H2O),                                                          # (mg/m²s) Apply heating correction to H2O flux when ambient temperature < TEMP_THR3
           LE_HC =  F_H2O_HC / 18 * 44.1, 
           F_CO2_CUM = (cumsum(coalesce(F_CO2, 0)) + F_CO2 * 0) * 30 * 60 * (12 / 44) / 1000,                                #  (mg/m²s) Cumulative FC before heating correction
           F_CO2_HC_CUM = (cumsum(coalesce(F_CO2_HC, 0)) + F_CO2_HC * 0) * 30 * 60 * (12 / 44) / 1000)                       #  (mg/m²s) Cumulative FC heating corrected
}

# adapted to Frank & Massmann (2020)
SSH_correction_Frank <- function(df_input, RAD_THR, K) {
  df_input %>%
    mutate(TEMP_K = TEMP_C + 273.15,                                                                                         # Air Temperature in K
           K_AIR =  0.000067 * TEMP_C + 0.024343,                                                                            # (W/mK) thermal conductivity coefficient of air
           L_TOP =  0.045,                                                                                                   # (m) Diameter of detector housing
           L_BOT =  0.065,                                                                                                   # (m) Diameter of source housing
           L_SPAR = 0.005,                                                                                                   # (m) Diameter of the spar
           D_TOP =  0.0028 * sqrt(L_TOP / MEAN_U) + 0.00025 / MEAN_U,                                                        # (m) Average thickness of boundary layer below top window
           D_BOT =  0.004  * sqrt(L_BOT / MEAN_U),                                                                           # (m) Average thickness of boundary layer above bottom window
           D_SPAR = 0.0058 * sqrt(L_SPAR / MEAN_U),                                                                          # (m) Average thickness of boundary layer around the spar
           T_TOP =  ifelse(DAYTIME == 1, (1.005 * TEMP_C + 0.24), (1.008 * TEMP_C - 0.41)) + 273.15,                         # (K) Surface temperature of top sensor housing (Burba, 2008), smoothed transition between night and day using tanh(K*RAD)
           T_BOT =  ifelse(DAYTIME == 1, (0.944 * TEMP_C + 2.57), (0.883 * TEMP_C + 2.17)) + 273.15,                         # (K) Surface temperature of bottom sensor housing (Burba, 2008), smoothed transition between night and day using tanh(K*RAD)
           T_SPAR = ifelse(DAYTIME == 1, (1.010 * TEMP_C + 0.36), (1.010 * TEMP_C - 0.17)) + 273.15,                         # (K) Surface temperature of spar (Burba, 2008), smoothed transition between night and day using tanh(K*RAD)
           S_TOP =  K_AIR * (((T_TOP - TEMP_K)  * (0.0225 + D_TOP)) / (0.0225 * D_TOP)),                                     # Sensible heat flux from top sensor housing 
           S_BOT =  K_AIR *  ((T_BOT - TEMP_K)  / D_BOT),                                                                    # (W/m²) Sensible heat flux from bottem sensor housing 
           S_SPAR = K_AIR *  ((T_SPAR - TEMP_K) / (0.0025 * log((0.0025 + D_SPAR) / 0.0025))),                               # (W/m²) Sensible heat flux from spar 
           HC =     (0.108 * S_BOT +  0.035 * S_TOP + 0.225 * S_SPAR) / RHOCP * CO2 / TEMP_K * (1 + 1.6077 * H2O / PD),      # (mg/m²/s) heating correction, add sensible heat fluxes from instrument surfaces to ambient sensible heat flux (eq. 4, Burba 2008), add weights from eq. 14, Frank & Massmann (2020)
           F_CO2 =  F_CO2 * 0.044,                                                                                           # (mg/m²s) F_CO2 before heating correction
           F_H2O =  F_H2O * 18,                                                                                              # (mg/m²s) F_H2O before heating correction
           F_CO2_HC = ifelse(TEMP_C < TEMP_THR, F_CO2 + HC, F_CO2),                                                          # (mg/m²s) Apply heating correction to CO2 flux when ambient temperature < TEMP_THR3
           F_H2O_HC = ifelse(TEMP_C < TEMP_THR, F_H2O + HC, F_H2O),                                                          # (mg/m²s) Apply heating correction to H2O flux when ambient temperature < TEMP_THR3
           LE_HC =  F_H2O_HC / 18 * 44.1,
           F_CO2_CUM = (cumsum(coalesce(F_CO2, 0)) + F_CO2 * 0) * 30 * 60 * (12 / 44) / 1000,                                #  (mg/m²s) Cumulative FC before heating correction
           F_CO2_HC_CUM = (cumsum(coalesce(F_CO2_HC, 0)) + F_CO2_HC * 0) * 30 * 60 * (12 / 44) / 1000)                       #  (mg/m²s) Cumulative FC heating corrected
}

# Burba correction --------------------------------------------------
df_SSH_corr_Burba <- SSH_correction_Burba(df_input = df_SSH_input, RAD_THR = RAD_THR, K = K)
df_SSH_corr_Frank <- SSH_correction_Frank(df_input = df_SSH_input, RAD_THR = RAD_THR, K = K)


# Figures for tuning ----------------------------------------------------------------
# Plot cummulative corrected and uncorrected F_CO2
df_SSH_corr <- df_SSH_corr_Frank

p1 <- df_SSH_corr %>%
  select(DATETIME, F_CO2_CUM, F_CO2_HC_CUM) %>%
  pivot_longer(cols = c(F_CO2_CUM, F_CO2_HC_CUM), names_to = "series", values_to = "FCO2") %>%
  ggplot(aes(DATETIME, FCO2, color = series)) +
  geom_line(na.rm=T, size = 2) +
  labs(x = "Time", y = expression('Cummulative CO'[2]*' flux [mg C m'^-2*']')) +
  scale_color_manual(labels = c("uncorrected", "corrected"),
                     values = c("firebrick", "dodgerblue")) +
  scale_x_datetime(date_breaks = "1 year", date_labels = "%Y")+
  theme_light() +
  theme(axis.title.x = element_blank(), panel.grid.major = element_line(linetype = "dashed"), panel.grid.minor = element_blank(),
        legend.title = element_blank(), legend.position = c(.2,.2))

# Plot daily mean corrected and uncorrected F_CO2
p2 <- df_SSH_corr %>%
  group_by(DOY) %>%
  summarise_at(.vars = c("TEMP_C", "HC", "F_CO2", "F_CO2_HC"), .funs = c("mean", "sd"), na.rm=T) %>%
  pivot_longer(cols = c(F_CO2_mean, F_CO2_HC_mean), names_to = "series", values_to = "FCO2") %>%
  mutate(series = factor(series, levels = c("F_CO2_mean", "F_CO2_HC_mean"))) %>%
  ggplot(aes(DOY, FCO2, color = series)) +
  geom_line(na.rm=T) +
  labs(x = "Time", y = expression('Daily mean CO'[2]*' flux [mg m'^-2*' s'^-1*']')) +
  scale_color_manual(labels = c("uncorrected", "corrected"),
                     values = c("firebrick", "dodgerblue")) + 
  scale_x_continuous(breaks = c(0,90,180,270,360)) +
  theme_light() +
  theme(axis.title.x = element_blank(), panel.grid.major = element_line(linetype = "dashed"), panel.grid.minor = element_blank(),
        legend.title = element_blank(), legend.position = c(.2,.2))

# Mean daily heating correction during cold conditions
p3 <- df_SSH_corr %>% 
  filter(TEMP_C < TEMP_THR) %>%
  group_by(DATE) %>% 
  summarise_at(.vars = c("TEMP_C", "HC", "F_CO2", "F_CO2_HC"), .funs = c("mean", "sd"), na.rm=T) %>%
  ggplot(aes(TEMP_C_mean, HC_mean)) +
  geom_point(na.rm=T) +
  labs(x = 'Temperature [°C]', y = expression('Daily mean heating correction [mg m'^-2*' s'^-1*']')) +
  theme_light() +
  theme(panel.grid.major = element_line(linetype = "dashed"), panel.grid.minor = element_blank(),
        legend.title = element_blank(), legend.position = c(.2,.2))

# monthly
p4 <- df_SSH_corr %>%
  mutate(F_CO2 = F_CO2 / 0.044,
         F_CO2_HC = F_CO2_HC / 0.044) %>%
  group_by(TIME_DEC, month(DATETIME)) %>%
  summarise_at(vars(F_CO2, F_CO2_HC), mean, na.rm = T) %>%
  pivot_longer(cols = c(F_CO2,  F_CO2_HC), names_to = "names", values_to = "values") %>%
  ggplot(aes(TIME_DEC, values, color = names))+
  geom_hline(yintercept = 0, color = "grey") +
  geom_line() +
  scale_color_manual(labels = c("uncorrected", "corrected"),
                     values = c("firebrick", "dodgerblue")) + 
  scale_x_continuous(breaks = c(0,6,12,18,24), labels = c(0,6,12,18,24)) +
  facet_wrap(~`month(DATETIME)`, ncol = 3, labeller = labeller(`month(DATETIME)` = month_name)) +
  labs(x = 'Hour of the day', y = expression('CO'[2]*' flux [mg m'^-2*' s'^-1*']')) +
  theme_light() +
  theme(panel.grid.major = element_line(linetype = "dashed"), panel.grid.minor = element_blank(),
        strip.background = element_rect(colour = "grey", fill = NA), strip.text = element_text(color = "black"),
        legend.title = element_blank(), legend.position = "none")

# annual
p5 <- df_SSH_corr %>%
  mutate(F_CO2 = F_CO2,
         F_CO2_HC = F_CO2_HC) %>%
  group_by(TIME_DEC, year(DATETIME)) %>%
  summarise_at(vars(F_CO2, F_CO2_HC), mean, na.rm = T) %>%
  pivot_longer(cols = c(F_CO2,  F_CO2_HC), names_to = "names", values_to = "values") %>%
  ggplot(aes(TIME_DEC, values, color = names))+
  geom_hline(yintercept = 0, color = "grey") +
  geom_line() +
  scale_color_manual(labels = c("uncorrected", "corrected"),
                     values = c("firebrick", "dodgerblue")) + 
  scale_x_continuous(breaks = c(0,6,12,18,24), labels = c(0,6,12,18,24)) +
  facet_wrap(~`year(DATETIME)`, ncol = 3) +
  labs(x = 'Hour of the day', y = expression('CO'[2]*' flux [mg m'^-2*' s'^-1*']')) +
  theme_light() +
  theme(panel.grid.major = element_line(linetype = "dashed"), panel.grid.minor = element_blank(),
        strip.background = element_rect(colour = "grey", fill = NA), strip.text = element_text(color = "black"),
        legend.title = element_blank(), legend.position = "none")



# matrix to set plot heights
lay <- rbind(c(1,1), c(2,2), c(3,4))

# arrange plots
SSH_plots <- arrangeGrob(p1, p2, p4, p5, layout_matrix = lay)

# export plots
ggsave(paste("SSH_plots_", dataset, "_Frank.png", sep = ""), width = 15, height = 12, SSH_plots, dpi = 300)

# export results ----------------------------------------------------------
df_SSH_corr_Burba_exp <- df_SSH_corr_Burba %>%
  select(F_CO2, HC, F_CO2_HC) %>%
  mutate_all(~./0.044) %>%
  rename(HC_Burba = HC,
         F_CO2_HC_Burba = F_CO2_HC)

df_SSH_corr_Frank_exp <- df_SSH_corr_Frank %>%
  select(HC, F_CO2_HC) %>%
  mutate_all(~./0.044) %>%
  rename(HC_Frank = HC,
         F_CO2_HC_Frank = F_CO2_HC)

df_SSH_corr_export <- cbind(df_openeddy, df_SSH_corr_Burba_exp, df_SSH_corr_Frank_exp)

df_SSH_corr_export %>%
  mutate(DATETIME = as.character(DATETIME)) %>%
  fwrite(file = paste("df_", dataset, "_openeddy_SSH.csv", sep = ""))



